/**
* Script: ship.js
* Written by: Radnen
* Updated: 1/16/2011
**/

function Stat(value, max) {
	if (value === undefined) value = max = 0;

	this.value = value;
	this.max = max;
	
	this.add = function(num) {
		if (this.value + num > this.max) return false;
		else {
			this.value += num;
			return true;
		}
	}
	
	this.remove = function(num) {
		if (this.value - num < 0) return false;
		else {
			this.value -= num;
			return true;
		}
	}
}

function ShipEquip(left, right, secondary, tech, shield) {
	this.left      = left;
	this.right     = right;
	this.tech      = tech;
	this.secondary = secondary;
	this.shield    = shield;
}

/**
* Base Ship Object:
**/
function ShipEntity()
{
	this.vx = 0; this.vy = 0;
	this.xx = 0; this.yy = 0;
	this.maxvx = 0; this.maxvy = 0;
	
	var jet_image = -1;
		
	this.name = "";
	this.shipname = "";

	this.target = null;
	this.target_a = new Angle();
	this.a = new Angle();
	
	var _x = 0, _y = 0;
	
	this.__defineGetter__("x", function() { return _x; });
	this.__defineGetter__("y", function() { return _y; });
	this.__defineGetter__("cx", function() { return _x + this.w/2; });
	this.__defineGetter__("cy", function() { return _y + this.h/2; });
	this.__defineSetter__("x", function(v) { _x = v - this.w/2; });
	this.__defineSetter__("y", function(v) { _y = v - this.h/2; });
	
	this.lastspeed = 5.0;
	this.maxspeed = 5.0;
	this.speed = 0;
	this.acc = 0.50; // acceleration
	this.last_a_msecs = 150;
	this.a_msecs = 150;
	this.r_msecs = 20;
	this.hp      = new Stat();
	this.escort  = new Stat();
	this.cap     = new Stat();
	this.shields = new Stat();
	this.jumps   = new Stat();
	
	this.shielded = false;
	this.shieldCol = CreateColor(255, 255, 255);
	this.shieldAlpha = new TimedValue(0, 500);
	
	this.isEnemy = false;
	this.thrust = false;
	
	/* Equipment: */
	this.equips = new ShipEquip(new Equip(-2), new Equip(-2), new Equip(-2), new Equip(-2), new Equip(-2));
	
	var a_time = 0;
	var r_time = 0;
	var rw_time = 0;
	var lw_time = 0;
	var t_time = 0;
	var s_time = 0;
	
	this.image = "shuttle";
	this.w = Resources.images[this.image].width;
	this.h = Resources.images[this.image].height;
	
	this.id = -1;
	this.canTurret = false;
	this.canBeam = false;
	this.jump = 0;
	this.jumpSys = null;
	this.targetSys = null;
	this.jumpTime = 0;
	this.onJump = function(){};
	this.afterJump = function(){};
	this.rating = 0;

	this.hyperJump = function(target) {
		if (this.jump != 0) return;
		this.jumpSys = GameWorlds[Genesis.currentWorld].origin;
		this.targetSys = target;
		this.lastspeed = this.maxspeed;
		this.maxspeed = 24;
		this.last_a_msecs = this.a_msecs;
		this.a_msecs = 25;
		this.target_a.radians = Math.atan2(this.targetSys.y-this.jumpSys.y, this.targetSys.x-this.jumpSys.x);
		this.jump = 1;
		this.jumpTime = 0;
		this.onJump();
	}
	
	this.drawShields = function() {	
		if (this.shielded)
			OutlinedCircle(this.cx+Camera.x, this.cy+Camera.y, this.w>>1, Colors.clearBlue, true);
		
		if (this.shieldAlpha.value > 0) {
			this.shieldCol.alpha = this.shieldAlpha.value;
			OutlinedCircle(this.cx+Camera.x, this.cy+Camera.y, this.w>>1, this.shieldCol, true); 
			this.shieldAlpha.update();
		}
	}
	
	this.hitShield = function(atk) {
		this.shieldAlpha.value = 255;
		this.shieldAlpha.scale(0);
		this.shields.value -= atk;
		if (this.shields.value < 0) this.shields.value = 0;
	}
	
	this.setupShip = function(obj) {
		// setup defaults: //
		this.shipname = "";
		this.acc = 0.50;
		this.maxspeed = 5.0;
		this.a_msecs = 150;	
		this.r_msecs = 20;
		this.canTurret = false;
		this.canBeam = false;
		this.shields.value = this.shields.max = 0;
		this.nameType = NAME_VESSEL;

		Utility.set(this, obj);
		for (var i in this.equips) {
			var eq = this.equips[i];
			if (eq.item >= 0) {
				if (eq.amount > 1) {
					Items[eq.item].equip(this);
					this.cap.value += Items[eq.item].weight*(eq.amount-1);
				}
				else Items[eq.item].equip(this);
			}
		}
		this.shields.value = this.shields.max;
		
		this.lastspeed = this.maxspeed;
		this.last_a_msecs = this.a_msecs;
		this.w = Resources.images[this.image].width;
		this.h = Resources.images[this.image].height;
	}
	
	this.doJump = function() {
		switch (this.jump) {
			case 1:
				if (this.jumpTime == 0 && this.turnToTarget()) {
					this.jumpTime = GetTime();
					Audio.playSound(Resources.sounds.teleport);
					this.vx = 0; this.vy = 0;
					this.wh_x = this.x + 4 + Math.cos(this.a.radians) * 3000;
					this.wh_y = this.y - 32 + Math.sin(this.a.radians) * 3000;
				}
				
				if (this.jumpTime + 3000 > GetTime()) {
					this.updateVelocity();
					Resources.images.wormhole.rotateBlit(this.wh_x+Camera.x, this.wh_y+Camera.y, this.a.radians);
				}
				else if (this.jumpTime != 0) {
					this.afterJump();
					this.jump = 2;
				}
			break;
			case 2:
				if (Math.floor(this.speed) > this.maxspeed) {
					this.updateVelocity();
					Resources.images.wormhole.rotateBlit(this.wh_x+Camera.x, this.wh_y+Camera.y, this.a.radians);					
				}
				else { this.a_msecs = this.last_a_msecs; this.jump = 0; }
			break;
		}
	}

	this.draw = function() {
		_x += this.vx;
		_y += this.vy;
		this.xx = Camera.x + _x;
		this.yy = Camera.y + _y;
		
		if (this.thrust && jet_image >= 0) Resources.images["jet_" + jet_image].rotateBlit(this.xx, this.yy, this.a.radians);
		Resources.images[this.image].rotateBlit(this.xx, this.yy, this.a.radians);
		this.drawShields();
	}
	
	this.drawRating = function() {
		var amt = Math.ceil(this.rating/4);
		while(amt--) Resources.images.starrating.blit(this.xx+8*amt, this.yy+this.h);
	}
		
	this.updateVelocity = function() {
		if (a_time + this.a_msecs < GetTime()) {
			this.maxvx = Math.cos(this.a.radians)*this.maxspeed;
			this.maxvy = Math.sin(this.a.radians)*this.maxspeed;
			var v_x = Math.abs(Math.cos(this.a.radians))*this.acc;
			var v_y = Math.abs(Math.sin(this.a.radians))*this.acc;
			if (this.maxvx > this.vx) this.vx += v_x;
			if (this.maxvx < this.vx) this.vx -= v_x;
			if (this.maxvy > this.vy) this.vy += v_y;
			if (this.maxvy < this.vy) this.vy -= v_y;
			this.speed = Math.sqrt(this.vx*this.vx + this.vy*this.vy);			
			
			this.animateThruster();
			a_time = GetTime();
		}
	}
	
	this.animateThruster = function() {
		if (t_time + 500 < GetTime()) {
			this.thrust = (this.speed > 0 && this.speed < this.maxspeed);
			if (jet_image < 2) jet_image++;
			t_time = GetTime();
		}
	}
	
	this.resetThrust = function() {
		this.thrust = false;
		jet_image = -1;
		t_time = 0;
	}
	
	this.turnLeft = function() {
		if (r_time + this.r_msecs < GetTime()) {
			this.a.degrees -= 5;
			r_time = GetTime();
		}
	}
	
	this.turnRight = function() {
		if (r_time + this.r_msecs < GetTime()) {
			this.a.degrees += 5;
			r_time = GetTime();
		}
	}
	
	this.impact = function(vx, vy) {
		this.vx += vx;
		this.vy += vy;
	}
	
	// image is a string representing the object identifier //
	this.setImage = function(image) {
		this.image = image;
		this.w = Resources.images[image].width;
		this.h = Resources.images[image].height;
	}
	
	this.firePrimary = function() {
		var single = (this.equips.left.item < 0 || this.equips.right.item < 0);
		if (this.equips.left.item  > -1 && lw_time + Items[this.equips.left.item].cooldown < GetTime()) {
			Items[this.equips.left.item].fire(this, (single) ? 0 : -8);
			lw_time = GetTime();
		}
		if (this.equips.right.item > -1 && rw_time + Items[this.equips.right.item].cooldown < GetTime()) {
			Items[this.equips.right.item].fire(this, (single) ? 0 :  8);
			rw_time = GetTime();
		}
	}
	
	this.fireSecondary = function() {
		if (this.equips.secondary.item > 0 && s_time + Items[this.equips.secondary.item].cooldown < GetTime()) {
			if (this.equips.secondary.amount > 0) {
				this.equips.secondary.amount--;
				this.cap.value -= Items[this.equips.secondary.item].weight;
				Items[this.equips.secondary.item].fire(this, 0);
			}
			s_time = GetTime();
		}
	}
	
	//this.updateConstant ...
	
	this.turnToTarget = function(target) {
		if (!target) target = this.target_a;
		
		var sd = Math.floor(this.a.degrees/5)*5;
		var td = Math.floor(target.degrees/5)*5;
		
		if (sd > td) { this.turnLeft(); return false; }
		else if (sd < td) { this.turnRight(); return false; }
		else { this.a.degrees = td; return true; }
	}
}

var Ships = [];

Ships[0] = ({
	shipname: "Shuttle",
	image: "shuttle",
	jumps: new Stat(2, 2),
	cap: new Stat(0, 30),
	escort: new Stat(0, 1),
	hp: new Stat(25, 25),
	equips: new ShipEquip(new Equip(0), new Equip(-2), new Equip(-2), new Equip(-2), new Equip(-2)),
	rating: 1
});

Ships[1] = ({
	shipname: "Taxi",
	image: "taxi_shuttle",
	maxspeed: 7.0,
	a_msecs: 120,
	r_msecs: 16,
	jumps: new Stat(4, 4),
	cap: new Stat(0, 45),
	escort: new Stat(0, 4),
	hp: new Stat(45, 45),
	equips: new ShipEquip(new Equip(0), new Equip(-1), new Equip(-2), new Equip(-1), new Equip(-1)),
	rating: 1,
});

Ships[2] = ({
	shipname: "Super Shuttle",
	image: "super_shuttle",
	maxspeed: 6.0,
	jumps: new Stat(3, 3),
	cap: new Stat(0, 45),
	escort: new Stat(0, 1),
	hp: new Stat(30, 30),
	equips: new ShipEquip(new Equip(0), new Equip(0), new Equip(-1), new Equip(-1), new Equip(-1)),
	rating: 2,
});

Ships[3] = ({
	shipname: "Caravan",
	image: "caravan",
	nameType: NAME_TRADER,
	maxspeed: 4.0,
	a_msecs: 165,
	jumps: new Stat(4, 4),
	cap: new Stat(0, 80),
	escort: new Stat(0, 3),
	hp: new Stat(45, 45),
	equips: new ShipEquip(new Equip(1), new Equip(-2), new Equip(-1), new Equip(-1), new Equip(7)),
	canTurret: true,
	rating: 2,
});

Ships[4] = ({
	shipname: "Mining Vessel",
	image: "mining_vessel",
	nameType: NAME_MINER,
	a_msecs: 140,
	r_msecs: 15,
	jumps: new Stat(3, 3),
	escort: new Stat(0, 1),
	cap: new Stat(0, 50),
	hp: new Stat(30, 30),
	equips: new ShipEquip(new Equip(3), new Equip(-1), new Equip(-1), new Equip(-2), new Equip(-1)),
	rating: 1,
});

Ships[5] = ({
	shipname: "Fighter",
	image: "fighter",
	nameType: NAME_MILITARY,
	maxspeed: 6.0,
	a_msecs: 125,
	r_msecs: 18,
	jumps: new Stat(4, 4),
	cap: new Stat(0, 55),
	escort: new Stat(0, 3),
	hp: new Stat(60, 60),
	equips: new ShipEquip(new Equip(0), new Equip(0), new Equip(5, 5), new Equip(-1), new Equip(7)),
	rating: 3,
});

Ships[6] = ({
	shipname: "Destroyer",
	image: "destroyer",
	nameType: NAME_MILITARY,
	maxspeed: 7.0,
	a_msecs: 140,
	r_msecs: 25,
	cap: new Stat(0, 75),
	jumps: new Stat(6, 6),
	escort: new Stat(0, 8),
	hp: new Stat(120, 120),
	equips: new ShipEquip(new Equip(1), new Equip(1), new Equip(6, 5), new Equip(-1), new Equip(8)),
	shields: new Stat(12, 12),
	rating: 4,
});